/*
	Generic BPR learning algorithm for tag recommendation.

	Based on the publication(s):
	Steffen Rendle, Christoph Freudenthaler, Zeno Gantner, Lars Schmidt-Thieme (2009): BPR: Bayesian Personalized Ranking from Implicit Feedback, in Proceedings of the 25th Conference on Uncertainty in Artificial Intelligence (UAI 2009)
	Steffen Rendle (2010): Context-aware Ranking with Factorization Models, in Series on Studies in Computational Intelligence, Springer, ISBN: 3642168973.

	Author:   Steffen Rendle, http://www.libfm.org/
	modified: 2010-12-10

	Copyright 2010 Steffen Rendle, see license.txt for more information
*/

#ifndef BPRLEARNER_H_
#define BPRLEARNER_H_

#include "Data.h"
#include "TagRecommender.h"

int LOSS_FUNCTION_SIGMOID = 0;
int LOSS_FUNCTION_LN_SIGMOID = 1;


class TagLearner {
	public:
		static inline double partial_loss(int loss_function, double x) {
			if (loss_function == LOSS_FUNCTION_SIGMOID) {
            			double sigmoid_tp_tn = (double) 1/(1+exp(-x));
              			return sigmoid_tp_tn*(1-sigmoid_tp_tn);
			} else if (loss_function == LOSS_FUNCTION_LN_SIGMOID) {
				double exp_x = exp(-x);
     				return exp_x / (1 + exp_x);
     			} else {
				assert((loss_function == LOSS_FUNCTION_LN_SIGMOID) || (loss_function == LOSS_FUNCTION_SIGMOID));	
			}             			
			return 0;
		}
		virtual void train(Dataset& dataset, TagRecommender& rec) = 0;	
			
};

class TagLearnerBPR : TagLearner{
	private:
		struct TagCase {
			int user_id;
			int item_id;
			int tag_id;
			const SparseVectorBoolean* post;
		};	
		int num_tag;	
		inline int drawTagNeg(Dataset& dataset, const SparseVectorBoolean* post);
	public:
		int num_iterations;
		int num_neg_samples;
		virtual void train(Dataset& dataset, TagRecommender& rec);	
};

void TagLearnerBPR::train(Dataset& dataset, TagRecommender& rec) {
	double total_time = getusertime();

	num_tag = dataset.max_tag_id + 1;
	
	std::cout << "Training BPR (Case-Update):"
			<< " num_iter=" << num_iterations
			<< " neg_samples=" << num_neg_samples
			<< std::endl;
			
	double f_best_f_measure = -1;
	

	// build tag case db:
	int num_tag_case = 0;
	for(SparseTensorBoolean::const_iterator t = dataset.data.begin(); t != dataset.data.end(); ++t) {
		for(SparseMatrixBoolean::const_iterator i = t->second.begin(); i != t->second.end(); ++i) {
			num_tag_case += i->second.size();
		}
	}

	TagCase* tag_case = new TagCase[num_tag_case];
	{
		int cntr = 0;
		for(SparseTensorBoolean::const_iterator t = dataset.data.begin(); t != dataset.data.end(); ++t) {
			for(SparseMatrixBoolean::const_iterator i = t->second.begin(); i != t->second.end(); ++i) {
				for(SparseVectorBoolean::const_iterator j = i->second.begin(); j != i->second.end(); ++j) {
					tag_case[cntr].user_id = t->first;
					tag_case[cntr].item_id = i->first;
					tag_case[cntr].post = & (i->second);
					tag_case[cntr].tag_id  = *j;
					cntr++;
				}
			}
		}
	}


	long long num_draws_per_iteration = num_tag_case * num_neg_samples;
		
	for (int iteration = 0; iteration < num_iterations; iteration++) {
    
		double iteration_time = getusertime();
		for (int draw = 0; draw < num_draws_per_iteration; draw++) {
			// draw the quadruple (u, i, t_p, t_n)
			int p  = rand() % num_tag_case;
			int u  = tag_case[p].user_id;
			int i  = tag_case[p].item_id;
			int tp = tag_case[p].tag_id;
			int tn = drawTagNeg(dataset, tag_case[p].post);
			rec.learn(u, i, tp, tn);    		
		}
		
		iteration_time = (getusertime() - iteration_time);
		std::cout << "Time: " << iteration_time << " / ";
		double this_f_measure = rec.evaluate(&dataset);
		f_best_f_measure = std::max(this_f_measure, f_best_f_measure);
		rec.auto_save(iteration+1);
	}
	delete [] tag_case;
	
	total_time = (getusertime() - total_time);
	std::cout << "training time: " << total_time << " s" << std::endl;	
	
}


inline int TagLearnerBPR::drawTagNeg(Dataset& dataset, const SparseVectorBoolean* post) {
	int t_n;
	do {
		t_n = rand() % num_tag;		
	} while (post->find(t_n) != post->end());
	return t_n;
}



#endif /*BPRLEARNER_H_*/
